/**
 * Axiom Anti-Detection Override
 * 
 * This script runs in the page context (MAIN world) and overrides
 * native browser APIs to prevent Axiom from detecting our extension.
 * 
 * MUST be injected at document_start before Axiom's scripts load.
 */

;(function () {
  'use strict'

  console.log('[Shuriken Blocker] Initializing Axiom overrides...')

  // =========================================================================
  // Part 1: MutationObserver Override
  // =========================================================================

  const nativeMutationObserver = window.MutationObserver

  /**
   * Override MutationObserver to neutralize detection callbacks
   * Checks callback source code for detection keywords
   */
  window.MutationObserver = class ShurikenMutationObserver extends nativeMutationObserver {
    constructor(callback, options) {
      // Inspect callback for detection patterns
      const callbackStr = callback.toString().toLowerCase()
      
      const isDetectionCallback =
        callbackStr.includes('visibility') || // Visibility detection
        callbackStr.includes('decoy') || // Decoy button detection
        callbackStr.includes('width') || // Dimension detection
        callbackStr.includes('eea7ed') // Our brand color

      if (isDetectionCallback) {
        // Neutralize the callback - replace with no-op
        console.log('[Shuriken Blocker] Blocked detection callback')
        callback = () => {}
      }

      super(callback, options)
    }
  }
  // No need to manually set prototype - class extends handles it automatically

  // =========================================================================
  // Part 2: WebSocket Interception
  // =========================================================================

  const nativeWebSocket = window.WebSocket
  const nativeAddEventListener = nativeWebSocket.prototype.addEventListener

  // Pulse data batching
  let pulseUpdateTimeout = null
  let pendingPulseData = []
  const DEBOUNCE_DELAY = 50 // ms
  const EXPIRY_TIME = 5e3 * 60 // 5 minutes in ms
  const CACHE_DURATION = 1000 // 1 second cache

  // Cached pulse data
  let cachedPulseData = null
  let cacheTimestamp = 0

  /**
   * Get cached pulse data (with 1 second cache)
   */
  function getCachedPulseData() {
    const now = Date.now()
    if (!cachedPulseData || now - cacheTimestamp > CACHE_DURATION) {
      try {
        const raw = localStorage.getItem('axiom.pulse')
        cachedPulseData = raw ? JSON.parse(raw) : { content: [] }
        cacheTimestamp = now
      } catch (error) {
        console.error('[Shuriken Blocker] Failed to read pulse cache:', error)
        cachedPulseData = { content: [] }
      }
    }
    return cachedPulseData
  }

  /**
   * Debounced pulse update to localStorage
   * Batches multiple updates to reduce write frequency
   */
  function debouncedPulseUpdate(newPulseData) {
    clearTimeout(pulseUpdateTimeout)

    pendingPulseData.push(newPulseData)

    pulseUpdateTimeout = setTimeout(() => {
      try {
        const allNewData = pendingPulseData.splice(0)
        const previousPulse = getCachedPulseData()

        const cutoffTime = Date.now() - EXPIRY_TIME
        const currentTimestamp = Date.now()

        // Build map of new tokens
        const newTokenMap = new Map()

        allNewData.forEach((pulseData) => {
          if (pulseData.content && Array.isArray(pulseData.content)) {
            pulseData.content.forEach((token) => {
              // Handle both array and object formats
              if (Array.isArray(token)) {
                newTokenMap.set(token[1], {
                  pairAddress: token[0],
                  tokenAddress: token[1],
                  lastSeen: currentTimestamp,
                })
              } else if (token.tokenAddress) {
                newTokenMap.set(token.tokenAddress, {
                  pairAddress: token.pairAddress,
                  tokenAddress: token.tokenAddress,
                  lastSeen: currentTimestamp,
                })
              }
            })
          }
        })

        // Filter expired previous data
        const filteredPrevious = (previousPulse.content || []).filter((token) => {
          const tokenTime =
            typeof token.lastSeen === 'number'
              ? token.lastSeen
              : new Date(token.lastSeen).getTime()
          return tokenTime > cutoffTime && !newTokenMap.has(token.tokenAddress)
        })

        // Merge old and new
        const mergedContents = [...filteredPrevious, ...newTokenMap.values()]

        const newPulseState = { content: mergedContents }
        localStorage.setItem('axiom.pulse', JSON.stringify(newPulseState))

        // Update cache
        cachedPulseData = newPulseState
        cacheTimestamp = Date.now()

        console.log(
          `[Shuriken Blocker] Pulse cache updated: ${mergedContents.length} tokens`
        )
      } catch (error) {
        console.error('[Shuriken Blocker] Error in batched pulse update:', error)
      }
    }, DEBOUNCE_DELAY)
  }

  /**
   * Check if WebSocket URL should be intercepted
   */
  function shouldInterceptWebSocket(url) {
    if (!url) return false
    const urlLower = url.toLowerCase()
    return urlLower.includes('axiom.trade')
  }

  /**
   * Override WebSocket constructor
   */
  window.WebSocket = function WebSocket(url, protocols) {
    const ws = new nativeWebSocket(url, protocols)

    // Only intercept Axiom WebSockets
    if (shouldInterceptWebSocket(url)) {
      console.log('[Shuriken Blocker] Intercepting Axiom WebSocket:', url)

      try {
        // Listen for pulse updates
        nativeAddEventListener.call(ws, 'message', function (event) {
          if (event && event.data && typeof event.data === 'string') {
            const eventDataLower = event.data.toLowerCase()
            
            // Pulse feed updates
            if (eventDataLower.includes('update_pulse')) {
              try {
                const newPulseData = JSON.parse(event.data)
                debouncedPulseUpdate(newPulseData)
              } catch (error) {
                console.error('[Shuriken Blocker] Error parsing pulse data:', error)
              }
            }
          }
        })

        // Auto-subscribe to pulse feed
        nativeAddEventListener.call(ws, 'open', function (event) {
          console.log('[Shuriken Blocker] WebSocket opened, subscribing to pulse feed')
          ws.send('{"action":"join","room":"update_pulse_v2"}')
        })
      } catch (error) {
        console.error('[Shuriken Blocker] Error adding message listeners:', error)
      }
    }

    return ws
  }

  // Preserve prototype chain
  window.WebSocket.prototype = nativeWebSocket.prototype
  window.WebSocket.prototype.constructor = window.WebSocket
  Object.setPrototypeOf(window.WebSocket, nativeWebSocket)

  console.log('[Shuriken Blocker] Overrides active ✓')
  
  // Set a clear marker for detection from content script
  window.__SHURIKEN_BLOCKER_ACTIVE__ = true
})()

